
------------------------------------------------------------------------------
-- Hotkey Manager -- Version 1.0
--
-- Utility script.
--
-- This script acts as a scripted interface for the hotkey management scripts
-- that hadn't yet received any sort of GUI.
-- written by : sean konrad (sean@eyeonline.com)
-- updated    : August 28, 2006.
-- 
------------------------------------------------------------------------------

-- Define the variable that points
-- to the hotkey manager object.

if not fusion then print ("ERROR: This script is not a command line script.  Don't try to run it as one.") do return end end

hKeyMan = fusion.HotkeyManager
hKeys = hKeyMan:GetHotkeys()

-- Basic sanity checks.
if not hKeyMan then print ("Error: Could not obtain hotkey manager object.\n  YOU SHOULD NOT SEE THIS ERROR...  EVER.") do return end end

globalmap = fusion:GetGlobalPathMap()

curCount = 0
unknownCount = 0

-- ensure eyeon.scriptlib is initialized.
ldofile(fusion:MapPath("Scripts:\\eyeon.scriptlib"))

-- Define Editor variable
-- this is currently unused but would theoretically
-- provide the user with an avenue for creating new scripts.
editor = fusion:GetPrefs("Global.Script.EditorPath")
if not fileexists(editor) then editor = "notepad" end

helpText = [[Contents:
	Introduction
	1) Categories
	2) Hotkey Names 
	3) How to Set Scripted Actions
	4) Adding and Removing Hotkeys
	5) Custom Actions
	
INTRODUCTION

Most hotkeys in Fusion are assigned using the internal scripting facilities of Fusion 
and defined in a Hotkeys.hotkey file stored in the profiles directory for the user 
(a subdirectory called "default" by default).  This script interprets that 
information and abstracts it into an interface.  The hotkeys are associated with various 
categories that are associated with various parts of the Fusion interface.  The Hotkeys 
themselves are assigned a name that is associated with the Microsoft hotkeys definitions.


1) Categories

As stated above, hotkeys are associated with various portions of the interface.  
Theses areas are defined below:  

	Global - This is a general category for the entire Fusion interface regardless of 
	composition context.
	
	Frame - This refers to the general interface associated with a composition window.  
	All hotkeys registered to work within the "Frame" context, will therefore work in 
	the Flow/Spline/Console/etc. views that are associated with composition, as opposed 
	to the global interface.
	
	View - Refers to the viewport that's currently selected.
	
	GLViewer - Refers to the Buffer currently active in the viewport (A/B View, 3D View).
	
	Preview - Refers to the preview buffer in the viewport
	
	Flow - The Flow View
	
	Spline Editor - The Spline Editor 
	
	Timeline - The Timeline Editor
	
	Console - The Console
	
	Info - The comments tab
	
	Bin - Refers to the Bin Window
	
	Controls - Refers to the tool controls window subsection of the frame.
	
2) Hotkey Names

Fusion uses the windows Hotkey definitions to deine its hotkey structure.  
For more information on this API go to: 
http://msdn.microsoft.com/library/en-us/winui/winui/WindowsUserInterface/UserInput/VirtualKeyCodes.asp .
In general, the hotkeys are fairly simple to understand: SHIFT_V indicates that you 
would hold the Shift Key and press the V key to activate the associated hotkey action.  
Some of the hotkeys are not clearly defined as they are associated with keys that are 
not universal to all keyboards (and therefore many have a prefix of "OEM").

3) How to Set Script Actions

The Hotkey Manager script defines a number of default actions available for the 
various categories directly within the script.  For example, if you click on the 
preview category item, you'll notice that Play/Pause 2 is associated with the 
"SPACE" hotkey.  If I wanted to change that, I would select the action I now wanted
the "SPACE" hotkey to be associated with from the Actions list, and then press 
"Set Action" (ensuring that "SPACE" was still selected in the tree view under the 
Preview branch).  

Additionally, these lists will be populated by any Custom Actions (see section 5) as 
well as composition scripts (in the Frame context) and utility scripts (in the Global 
context).  Those actions that are not defined by the default actions or in the various 
custom / script action lists gathered by the script on launch.

4) Adding and Removing Hotkeys

To add a hotkey, go to the file menu at the top of the manager, Add Hotkey, select the 
category from the dropdown menu, and select the hotkey name based on the Windows hotkey 
definitions.  Then press "Add".   This will add the hotkey to that category.  It will 
come in set to having "no action".

To remove a hotkey, right click on the leaf in the tree view, and select Remove Hotkey.

5) Custom Actions

The first time the script is launched, Fusion will create directory in the area mapped
for Scripts:\ in the path mapping preferences called "Hotkeys".  If you wish to create
custom actions to populate the actions list, put them here.  For example, if one wanted
to define a shortcut that printed "Hello World" to the console, we could make a script
called "Hello World.eyeonscript" and place it in Scripts:\Hotkeys\ directory.  The
script's contents would be:

print("Hello World")
]]

-- Function that will rip apart the hKeys table
-- into a valid lua .hotkey file.
function hotkeysToString(tVal, stringWritePass,numTabs)
	local stringWrite = stringWritePass or "{"
	numTabs = numTabs or 1

	local i, v
	for i, v in pairs(tVal) do
		if tonumber(i) then
			placeHold = "[\""..tostring(i).."\"]"
		else
			placeHold = i
		end
		if type(v) == "table" then
			stringWrite = stringWrite.."\n"..string.rep("\t",numTabs)..placeHold.. " = "..hotkeysToString(v,"{",numTabs+1)..","
		elseif type(v) == "string" then
			stringWrite = stringWrite.."\n"..string.rep("\t",numTabs)..placeHold.." = ".."\""..v.."\""..","
		end
	end
	stringWrite = stringWrite.."\n"..string.rep("\t",numTabs-1).."}"
	
	return stringWrite
end

-- Check to see if the function is in our list of assembled functions
-- which we find to be just lovely.
function checkFunction(t_CrossCheck,item)
	-- Trim trailing spaces.
	
	item = eyeon.trim(item)
	-- Iterate through the table that we're checking.
	local k, v
	for k, v in pairs(t_CrossCheck) do
		if v == item then
			-- return true if the function is found in the passed array.
			return true
		end
		
	end
end

-- Function that gathers errant functions in order to preserve things.
function gatherUnknownFunctions()
	-- let's iterate through our table.
	local k, v
	for k, v in pairs(parallel_t) do
		
		-- see if it exists in any of the tables.
		if checkFunction(list_funcs["Frame"],v) or checkFunction(list_funcs["View"],v) or checkFunction(list_funcs["GLViewer"],v) or 
			checkFunction(list_funcs["Preview"],v) or checkFunction(list_funcs["Timeline"],v) or checkFunction(list_funcs["SplineEditor"],v) or
			checkFunction(list_funcs["LUTFrame"],v) or checkFunction(list_funcs["Global"],v) or  checkFunction(list_funcs["LUTMacroFrame"],v) or 
			checkFunction(list_funcs["Flow"],v) or checkFunction(list_funcs["LUTView"],v) or checkFunction(list_funcs["Info"],v) or
			checkFunction(list_funcs["Bin"],v) or checkFunction(list_funcs["Info"],v) or checkFunction(list_funcs["Console"],v) or checkFunction(list_funcs["Controls"],v) or
			checkFunction(list_funcs["GL3DViewer"],v) or checkFunction(list_funcs["Manager"],v) or checkFunction(list_funcs["Chat"],v) or
			checkFunction(list_funcs["GLImageViewer"],v) or checkFunction(scriptList,v) or string.sub(v, 1, 1) == "@"
		then
		else
			-- Don't initialize the variable if we don't need it.
			if not unknownFunctions then unknownFunctions = {} unknownCount = 0 end
			-- index and store these errant functions.
			unknownCount = unknownCount +1
			unknownFunctions["UnknownAction"..unknownCount] = v 
		end
	end
end

-- Function that gets the literal scripted action
-- for the text box below the list box.
function DisplayScript2(nodeParentName,hKey_Action)
	return list_funcs[nodeParentName][hKey_Action]
end

-- Necessary function that will select the right element when
-- a new item is selected in the tree.
function SelectRightItem(nodeName,nodeParentName)
	-- Check to make sure the table is initialized and 
	-- make sure they haven't clicked the parent recently..
	if not hKeys[nodeParentName] then
		return
	end
	
	-- Get the Action
	hKey_Action = hKeys[nodeParentName][nodeName]
	
	-- Iterate through the function list
	if list_funcs[nodeParentName] then
		local k, v
		for k, v in pairs(list_funcs[nodeParentName]) do
			-- If that's the associated action
			local trimmed = eyeon.trim(hKey_Action)
			if v == trimmed then
				-- store the name of the action
				index_key = k
				break
			elseif string.sub(trimmed,1,1) == "@" then
				-- script action, let's be more flexible about the path
				local path = string.gsub(trimmed,":/",":")
				if v == path then
					index_key = k
					break
				end
			end
		end
		
		-- now iterate through the alphabetized list.
		for k, v in pairs(list_alpha[nodeParentName]) do
		
			-- see if the value of the numeric key
			-- is the same
			if v == index_key then
				-- highlight that item in the list box
				-- seeing as the alpha_funcs list is a parallel
				-- array to the list box's array
				listboxScripts.value = tostring(k)
				-- set the text box.
				textScript.value = hKey_Action
				break
			end
		end
	end
end


-- Function to clear the Scripts ListBox as well as ye olde 
-- text box.
function ClearListbox(listbox, count)
	local i
	for i = 1, count do
		listbox[tostring(i)] = nil
	end
	textScript.value = ""
	return 0
end

-- Functions that will display the proper list box information.
function DisplayInformations(nodeName, nodeParentName) 
	
	-- we need to build an appropriate list
	-- for each item in the tree control.
	-- the list will be assembled based on category.
	-- all scripted actions are assigned to a category
	-- of common actions (view related functions in the view
	-- table, global in the globas) as well as actions
	-- defined in the Hotkey folder, and then a third set for
	-- errant functions (so that they're not lost).
	-- In order to prevent our lists from constantly being rebuilt 
	-- every single time a user clicks on a tree item to see one's 
	-- hotkeys or to assign it to an action, we check to see
	-- if the parent is the same as the previously selected parent.
	if prevParent == nodeParentName then
		-- make sure they just didn't select the same item twice.
		if prevNode ~= nodeName then
			-- Call the function to select the currently assigned action.
			SelectRightItem(nodeName,nodeParentName)
		end
	else
	
		-- Check to see if the parent is "Hotkeys"
		-- which is the root.  If it is, then we need to clear the lists
		
		if nodeParentName ~= "Hotkeys" then
			
			-- reset the count and clear the list box
			curCount = ClearListbox(listboxScripts,curCount)
			-- reset another count
			listCount = 0
			
			-- iterate through list_alpha of the parent.
			for k, v in pairs(list_alpha[nodeParentName]) do
				
				-- keep a count of the number of elements in this table while it's iterating 
				-- up.  
				curCount = curCount + 1
				listCount = listCount + 1
				
				-- Assign text to the list box element.
				listboxScripts[tostring(listCount)]=v
			end
			-- Call the SelectRightItem function..
			SelectRightItem(nodeName,nodeParentName)
		else
			-- OR CLEAR THE LIST/TEXT
			curCount = ClearListbox(listboxScripts,curCount)
		end
	end
	
	-- YES
	prevParent = nodeParentName
	prevNode = nodeName
end

-- Function that will set the currently selected function to nil.
function removeHotkey()
	-- Get the parent and cur item...
	local id = tree.value
	local name = tree["NAME"..tostring(id)]
	local parent = tree["PARENT"..tostring(id)]
	local parent = tree["NAME"..tostring(parent)]

	-- Pop a warning using Alarm!
	local answer = iup.Alarm("Warning","Are you Sure you want to remove this hotkey?  \n"
		.."If the Hotkey is part of the default Fusion\n"
		.."set, then it will reset to default after clicking \"Yes\".",
		"Yes","Cancel")
	if answer == 1 then
		hKeys[parent][name] = nil
		tree.delnode = "MARKED"
		
		-- Instead of being called from a method, redrawing the tree requires 
		-- setting a parameter.
		tree.redraw = "YES"
	end
end



--------------------------------------------------------------------------------
-- COMMON SCRIPT ACTIONS
--------------------------------------------------------------------------------
do 
	local t

	globals.list_funcs = {}
	
	list_funcs["Frame"] = {}
	t = list_funcs["Frame"]
	t["Jump to Render Start"] = "self.Composition.CurrentTime = self.Composition:GetAttrs().COMPN_RenderStart"
	t["Jump to Global Start"] = "self.Composition.CurrentTime = self.Composition:GetAttrs().COMPN_GlobalStart"
	t["Jump to Render End"] = "self.Composition.CurrentTime = self.Composition:GetAttrs().COMPN_RenderEnd"
	t["Jump to Global End"] = "self.Composition.CurrentTime = self.Composition:GetAttrs().COMPN_GlobalEnd"
	t["Load Settings 1"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(1) end"
	t["Load Settings 2"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(2) end"
	t["Load Settings 3"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(3) end"
	t["Load Settings 4"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(4) end"
	t["Load Settings 5"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(5) end"
	t["Load Settings 6"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(6) end"
	t["Load Settings 7"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(7) end"
	t["Load Settings 8"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(8) end"
	t["Load Settings 9"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(9) end"
	t["Load Settings 0"] = "t = self.Composition.ActiveTool; if t then t:SetCurrentSettings(10) end"
	t["Play/Pause"] = "if self.Composition:IsPlaying() then self.Composition:Stop() else self.Composition:Play() end"
	t["Redo"] = "self.Composition:Redo()"
	t["Undo"] = "self.Composition:Undo()"
	t["Save Settings 1"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.1]], t:SaveSettings(false)) end"
	t["Save Settings 2"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.2]], t:SaveSettings(false)) end"
	t["Save Settings 3"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.3]], t:SaveSettings(false)) end"
	t["Save Settings 4"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.4]], t:SaveSettings(false)) end"
	t["Save Settings 5"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.5]], t:SaveSettings(false)) end"
	t["Save Settings 6"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.6]], t:SaveSettings(false)) end"
	t["Save Settings 7"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.7]], t:SaveSettings(false)) end"
	t["Save Settings 8"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.8]], t:SaveSettings(false)) end"
	t["Save Settings 9"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.9]], t:SaveSettings(false)) end"
	t["Save Settings 0"] = "t = self.Composition.ActiveTool; if t then t:SetData([[Settings.10]], t:SaveSettings(false)) end"
	t["View On Left"] = "self:ViewOn(self.Composition.ActiveTool, 1)"
	t["View On Right"] = "self:ViewOn(self.Composition.ActiveTool, 2)"
	t["View On 3"] = "self:ViewOn(self.Composition.ActiveTool, 3)"
	t["View On 4"] = "self:ViewOn(self.Composition.ActiveTool, 4)"
	t["View On 5"] = "self:ViewOn(self.Composition.ActiveTool, 5)"
	t["View On 6"] = "self:ViewOn(self.Composition.ActiveTool, 6)"
	t["View On 7"] = "self:ViewOn(self.Composition.ActiveTool, 7)"
	t["View On 8"] = "self:ViewOn(self.Composition.ActiveTool, 8)"
	t["View On 9"] = "self:ViewOn(self.Composition.ActiveTool, 9)"
	t["Stop"] = "self.Composition:Stop()"
	t["Play/Pause 3"] = "if self.Composition:IsPlaying() then self.Composition:Stop() else self.Composition:Play(true) end"
	t["Jog forward by Frame Step Amount"] = "self.Composition.CurrentTime = self.Composition.CurrentTime + self.Composition:GetPrefs([[Comp.Transport.FrameStep]])"
	t["Jog back by Frame Step Amount"] = "self.Composition.CurrentTime = self.Composition.CurrentTime - self.Composition:GetPrefs([[Comp.Transport.FrameStep]])"
	t["Clear Frame's Views"] = "self:ViewOn()"
	t["Jump To Next Keyframe"] = "self.Composition.CurrentTime = self.Composition:GetNextKeyTime(self.Composition.CurrentTime, self.Composition.ActiveTool)"
	t["Jump To Previous Keyframe"] = "self.Composition.CurrentTime = self.Composition:GetPrevKeyTime(self.Composition.CurrentTime, self.Composition.ActiveTool)"
	t["Stop Render"] = "self.Composition:Stop(); self.Composition:AbortRenderUI()"
	t["No Action"] = ""
	
	list_funcs["View"] = {}
	t = list_funcs["View"]
	t["Lock/Unlock"] = "self:SetLocked(not self:GetLocked())"
	t["Pass Through Current Tools"] = "self:DisableCurrentTools()"
	t["Turn Quad View On/Off"] = "self:ShowQuadView(not self:ShowingQuadView())"
	t["Reset View"] = "self:ResetView()"
	t["Set A View"] = "self:SetBuffer(0)"
	t["Set B View"] = "self:SetBuffer(1)"
	t["Set A/B View"] = "self:SetBuffer(2)"
	t["Show/Hide Subview"] ="self:ShowSubView(not self:ShowingSubView())"
	t["Swap Subview with Main View"] = "self:SwapSubView()"
	t["Clear View"] = "self:GetPreview(0):ViewOn(NULL); self:GetPreview(1):ViewOn(NULL); self:SetBuffer(0)"
	t["Set View to Fit"] = "self:SetScale(0)"
	t["Zoom In"] = "self:SetScale(self:GetScale() * 1.4142135623730950488016887242097)"
	t["Zoom Out"] = "self:SetScale(self:GetScale() * 0.70710678118654752440084436210485)"
	t["No Action"] = ""
	
	list_funcs["Global"] = {}
	t = list_funcs["Global"]
	t["Load Hotkeys"] = "fusion.HotkeyManager:LoadHotkeys()"
	t["Purge Cache Manager"] = "fusion.CacheManager:Purge()"
	t["Show/Hide Render Manager"] = "fusion:ToggleRenderManager()"
	t["Show/Hide Bins"] = "fu:ToggleBins()"
	t["Show Console"] = "eyeon.allocconsole()"
	t["No Action"] = ""
	
	list_funcs["LUTFrame"] = {}
	t = list_funcs["LUTFrame"]
	t["Redo"] = "self.Composition:Redo()"
	t["Undo"] = "self.Composition:Undo()"
	t["No Action"] = ""
	
	list_funcs["LUTMacroFrame"] = {}
	t = list_funcs["LUTMacroFrame"]
	t["Redo"] = "self.Composition:Redo()"
	t["Undo"] = "self.Composition:Undo()"
	t["No Action"] = ""
	
	list_funcs["LUTView"] = {}
	t = list_funcs["LUTView"]
	t["Zoom In"] = "self:ZoomIn()"
	t["Zoom Out"] = "self:ZoomOut()"
	t["Zoom to Fit"] = "self:ZoomFit()"
	t["Zoom to Rectangle"] = "self:ZoomRectangle()"
	t["No Action"] = ""
	
	list_funcs["Timeline"] = {}
	t = list_funcs["Timeline"]
	t["Zoom In"] = "self:ZoomIn()"
	t["Zoom Out"] = "self:ZoomOut()"
	t["Zoom to Fit"] = "self:ZoomFit()"
	t["Zoom to Rectangle"] = "self:ZoomRectangle()"
	t["Jump to Previous Key Time"] = "self:GoPrevKeyTime()"
	t["Jump to Next Key Time"] = "self:GoNextKeyTime()"
	t["No Action"] = ""
	
	list_funcs["SplineEditor"] = {}
	t = list_funcs["SplineEditor"]
	t["Zoom In"] = "self:ZoomIn()"
	t["Zoom Out"] = "self:ZoomOut()"
	t["Zoom to Fit"] = "self:ZoomFit()"
	t["Zoom to Rectangle"] = "self:ZoomRectangle()"
	t["Jump to Previous Key Time"] = "self:GoPrevKeyTime()"
	t["Jump to Next Key Time"] = "self:GoNextKeyTime()"
	t["No Action"] = ""
	
	list_funcs["Preview"] = {}
	t = list_funcs["Preview"]
	t["Jog Back 10 Frames"] = "self:Seek(self:GetAttrs().PV_CurrentFrame - 10)"
	t["Jog Forward 10 Frames"] = "self:Seek(self:GetAttrs().PV_CurrentFrame + 10)"
	t["Frame Forward"] = "self:Seek(self:GetAttrs().PV_CurrentFrame + 1)"
	t["Frame Back"] = "self:Seek(self:GetAttrs().PV_CurrentFrame - 1)"
	t["Jog to Last Frame"] = "self:Seek(self:GetAttrs().PV_OutFrame - 1)"
	t["Jog to First Frame"] = "self:Seek(self:GetAttrs().PV_InFrame)"
	t["Play/Pause"] = "if self:IsPlaying() then self:Stop() else self:Play(true) end"
	t["Play/Pause 2"] = "if self:IsPlaying() then self:Stop() else self:Play() end"
	t["No Action"] = ""
	
	list_funcs["GLViewer"] = {}
	t = list_funcs["GLViewer"]
	t["Switch to Alpha Channel"] = "self:SetChannel(3); self:Redraw()"
	t["Switch to Red Channel"] = "self:SetChannel(0); self:Redraw()"
	t["Switch to Green Channel"] = "self:SetChannel(1); self:Redraw()"
	t["Switch to Blue Channel"] = "self:SetChannel(2); self:Redraw()"
	t["Switch to Color Channel"] = "self:SetChannel(-1); self:Redraw()"
	t["Switch to Z Channel"] = "self:SetChannel(12); self:Redraw()"
	t["Show/Hide View Controls"] = "self:ShowControls(not self:AreControlsShown()); self:Redraw()"
	t["Show/Hide View Guides"] = "self:ShowGuides(not self:AreGuidesShown()); self:Redraw()"
	t["No Action"] = ""

	list_funcs["GLImageViewer"] = {}
	t = list_funcs["GLImageViewer"]
	t["Toggle RoI"] = "self:EnableRoI(); self:Redraw()"
	t["No Action"] = ""

	list_funcs["GL3DViewer"] = {}
	t = list_funcs["GL3DViewer"]
	t["Center Selected"] = "self:CenterSelected(); self:Redraw()"
	t["Fit Selected"] = "self:FitSelected(); self:Redraw()"
	t["Fit All"] = "self:FitAll(); self:Redraw()"
	t["No Action"] = ""

	list_funcs["Flow"] = {}
	list_funcs["Flow"]["No Action"] = ""

	list_funcs["Bin"] = {}
	list_funcs["Bin"]["No Action"] = ""

	list_funcs["Info"] = {}
	list_funcs["Info"]["No Action"] = ""

	list_funcs["Console"] = {}
	list_funcs["Console"]["No Action"] = ""

	list_funcs["Controls"] = {}
	list_funcs["Controls"]["No Action"] = ""

	list_funcs["Manager"] = {}
	list_funcs["Manager"]["No Action"] = ""

	list_funcs["Chat"] = {}
	list_funcs["Chat"]["No Action"] = ""
end


--------------------------------------------------------------------------------
-- CUSTOM SCRIPT ACTIONS
--------------------------------------------------------------------------------


function ReadScriptDir(tbl, path, createdirs)
	local mp = MultiPath(path)
	
	mp:Map(globalmap)
	
	-- Create dirs if they don't exist
	if createdirs then
		for i,v in ipairs(mp:GetSegmentTable()) do
			if not direxists(v) then
				createdir(v)
			end
		end
	end
	
	local dir = mp:ReadDir("*.eyeonScript", true, true)
	
	for i,v in ipairs(dir) do
		tbl[v.RelativePath] = "@" .. path .. v.RelativePath
	end
end

function populateFunctionList()
	-- Make everything available for Comp scripts available in the Frame view.
	ReadScriptDir(list_funcs["Frame"], "Scripts:Comp/")
	
	-- Make everything that's available in the Utility directory available in the Global view.
	ReadScriptDir(list_funcs["Global"], "Scripts:Utility/")
	
	-- Make everything that's available in the Tool directory available in the Controls view.
	ReadScriptDir(list_funcs["Controls"], "Scripts:Tool/")
	ReadScriptDir(list_funcs["Flow"], "Scripts:Tool/")
	ReadScriptDir(list_funcs["View"], "Scripts:Tool/")
	
	-- Make everything that's available in the Bin directory available in the Bin view.
	ReadScriptDir(list_funcs["Bin"], "Scripts:Bin/")
	
	-- Get any additional scripts that are made available in the HotKeyScripts directory.
	-- An addition to keep menu items clean.
	globals.scriptList = {}
	ReadScriptDir(globals.scriptList, "Scripts:HotkeyScripts/", true)
end

populateFunctionList()

--------------------------------------------------------------------------------
-- DEFINING THE HOTKEY LIST OPTIONS
--------------------------------------------------------------------------------
table_Hotkeys = hKeyMan:GetKeyNames()


function populateTree(rebuild, parent)
	if not rebuild then
		-- Get a list of all hotkeys
		hKeys = hKeyMan:GetHotkeys()
	end
	t = {}
	parallel_t = {}
	-- We should be able to automatically structure
	-- the branches and leaves based on the table strucutre
	-- of the hotkeys.  The key index of the hKeys table will define 
	-- the context of the hotkeys.
	local tCount = 0
	local gCount = 0
	
	local k,v, i
	for k, v in pairs(hKeys) do
		-- the variable in v (assigned in the for
		-- idiom) will always equal a table.  so we
		-- should iterate through it.
		tCount = tCount +1
		t[tCount] = {}
		t[tCount].branchname = k
		local lCount = 0
		for i, hKey in pairs(v) do
			lCount = lCount + 1
			gCount = gCount + 1
			t[tCount][lCount] = i
			parallel_t[gCount] = hKey
			--parallel_t[tCount][lCount][i] = hKey
		end
	end
	for k, v in pairs(t) do
		table.sort(v, function (a,b) return (b > a) end)
	end
	table.sort(t, function (a,b) return (b.branchname > a.branchname) end)
	if not rebuild then
		iup.TreeSetValue(tree,t)
	else
		tree.value = "ROOT"
		tree.delnode = "CHILDREN"
		
		iup.TreeSetValue(tree,t)
		
		tree.redraw = "YES"
		
		--tree.value = tree[]
		--tree.addleaf()
	end
end

list_alpha = {}
-- ALPHABETIZE THE LISTS
function ChangeListsToAlphabetical()	
	local k,v, j, x
	for k, v in pairs(hKeys) do
		list_alpha[k] = {}
		local count = 0

		if list_funcs[tostring(k)] ~= nil then
			for j,x in pairs(list_funcs[tostring(k)]) do
				count = count +1 
				list_alpha[k][count] =  j
			end
			table.sort(list_alpha[k], function (a,b) return (b > a) end)
		end
	end
end
-------------------------------------------------------
-- Chunk for combining lists for faster compiling.
-------------------------------------------------------
function StitchLists(local_list)
	local k,v
	for k, v in pairs(scriptList) do
		local_list[k] = v
	end
	if unknownFunctions then
		for k, v in pairs(unknownFunctions) do
			local_list[k] = v
		end
	end
end

function ConcatenateLists()
	local k
	for k in pairs(hKeys) do
		if list_funcs[k] then
			StitchLists(list_funcs[k])
		end
	end
end

--------------------------------------------------------------------------------
-- INTERFACE RELATED CODE
--------------------------------------------------------------------------------

do -- Conceptual chunk

	-- Manufacture our tree view.
	tree = iup.tree{SIZE="200x100", EXPAND="VERTICAL", BGCOLOR="60 60 60", FGCOLOR = "200 200 200"}
	-- Create the list boxes and the text input that we'll use
	-- to define hotkeys and their associated scripting actions
	listboxScripts = iup.list{SIZE = "200x50", EXPAND="YES" }
	textScript = iup.multiline{SIZE = "200x50", EXPAND="YES" }
	listboxHotkeys = iup.list{}
	
	-- Button to set the action
	btnSetScript = iup.button{ title = "Set Action", SIZE="200x", EXPAND="HORIZONTAL", FLAT="YES"}
	
	-- Button to apply
	btnApply = iup.button{ title = "OK", size = 100, FLAT = "YES"}
	
	-- Button to exit.
	btnExit = iup.button{ title = "Exit",size=100,FLAT="YES"}
	
	-- Creates menu displayed when the right mouse button is pressed
	DeleteHotkey = iup.item {title = "Delete Hotkey"}
	AddNewHotkey = iup.item {title = "Add New Hotkey"}
	
	-- Context Menu Item.
	contextDelMenu = iup.menu{DeleteHotkey, AddNewHotkey}
	contextAddMenu = iup.menu{AddNewHotkey}

	-- File Menu
	do 
		item_DeleteHotkey = iup.item {title = "Delete Hotkey", key = "D"}
		item_AddHotkey    = iup.item {title = "Add Hotkey", key = "A"}
		item_Help = iup.item {title = "Help", key = "Help"}
		
		-- item_CreateCustomAction = iup.item {title = "Create Custom Action", key = "C"}
		
		-- Disabled feature -- eventually might be in.  Theoretically we should
		-- also have some ability to create a set of hotkeys that a user can download.
		-- Therefore we'll write out any of the "unknown functions", the scripts in the 
		-- Hotkeys directory, and then also any other relevant data.
		
		--item_ImportHotkeys = iup.item {title = "Import Hotkeys", key = "I"}
		--item_ExportHotkeys = iup.item {title = "Export Hotkeys", key = "E"}
		
		-- Creates two menus
		menu_file = iup.menu {item_DeleteHotkey, item_AddHotkey, item_Help}
		
		submenu_file = iup.submenu {menu_file, title = "File"}
		menu = iup.menu {submenu_file}
		
		function item_AddHotkey:action() 
			local id = tree.value
			local parent = tree["NAME"..id]
			if tree["KIND"..id] == "LEAF" then
				parent = tree["PARENT"..id]
				parent = tree["NAME"..parent]
			end

			setListCategory(parent)
			dlg_hKeyAdd.PARENTDIALOG = dlg
			dlg_hKeyAdd:showxy(iup.CENTER,iup.CENTER)
			buildHKeyBox(parent)
		end
		function item_DeleteHotkey:action() 
			removeHotkey();
		end
		
		do
			textHelp = iup.multiline{SIZE = "400x200", READONLY = "YES", WRAP="YES"}
			textHelp.value = helpText
	
			
			dlg_Help  = iup.dialog{iup.vbox{textHelp},title="Help",BGCOLOR="60 60 60", FGCOLOR = "200 200 200", TOPMOST = "NO",  RESIZE = "NO", MAXBOX = "NO"}
			function item_Help:action() 
				dlg_Help:showxy(iup.CENTER,iup.CENTER)
			end
		end
		
		-- Adding some sugar to the script that will launch an editor with a new script 
		-- where a person will be able to define their own custom action.
		-- Disabled.
		--~ 		function item_CreateCustomAction()
		--~ 				

		--~ 		end
	end
	
	
	
	-- Callback of the selection of an item.
	function tree:selection_cb(id)
		
		if not id then do return end end
		prev_ID = id
		--tree.value = id
		name = tree["NAME"..tostring(id)]
		parent = tree["PARENT"..tostring(id)]		
		parent = tree["NAME"..tostring(parent)]
		DisplayInformations(name,parent)
		
		-- menu:popup(iup.MOUSEPOS,iup.MOUSEPOS)
		return iup.DEFAULT
		
	end
	
	-- We want to pop up a contextual menu when right clicking on the tree, so this is the 
	-- way to do it.
	function tree:rightclick_cb(id)
		tree.value = id
		if tree["KIND"..id] == "LEAF" then
			contextDelMenu:popup(iup.MOUSEPOS,iup.MOUSEPOS)
		else
			contextAddMenu:popup(iup.MOUSEPOS,iup.MOUSEPOS)
		end
		return iup.DEFAULT
	end

	function tree:k_any(key)
		if key == K_DEL then
			removeHotkey()
		end
	end
	
	-- action for deleting the key -- that is, when the contextual menu is clicked.
	function DeleteHotkey:action()
		-- broken into a separate function in case a button is added at some point.
		removeHotkey()
	end
	function AddNewHotkey:action()
		local id = tree.value
		local parent = tree["NAME"..id]
		if tree["KIND"..id] == "LEAF" then
			parent = tree["PARENT"..id]
			parent = tree["NAME"..parent]
		end

		setListCategory(parent)
		dlg_hKeyAdd.PARENTDIALOG = dlg
		dlg_hKeyAdd:showxy(iup.CENTER,iup.CENTER)
		buildHKeyBox(parent)
	end
	
	function listboxScripts:action()
		id = tree.value
		name = tree["NAME"..tostring(id)]
		parent = tree["PARENT"..tostring(id)]
		parent = tree["NAME"..tostring(parent)]
		
		-- Display the script.  Using this function.  To do so.
		textScript.value = DisplayScript2(parent,listboxScripts[listboxScripts.value])	
	end
	
	-- Function to set the action of the interface.
	function btnSetScript:action()
		id = tree.value
		name = tree["NAME"..tostring(id)]
		parent = tree["PARENT"..tostring(id)]
		parent = tree["NAME"..tostring(parent)]

		v = listboxScripts.value
		scriptAction = listboxScripts[tostring(v)]
		
		-- has the user changed the script?
		if textScript.value ~= list_funcs[parent][scriptAction] then
			hKeys[parent][name] = textScript.value

			local i
			for i = 1,1000 do
				if listboxScripts[tostring(i)] == nil then
					unknownCount = unknownCount + 1
					scriptAction = "UnknownAction"..unknownCount
					
					list_funcs[parent][scriptAction] = textScript.value
					listboxScripts[tostring(i)] = scriptAction
					listboxScripts.value = tostring(i)
					break
				end
			end
		else
			hKeys[parent][name] = list_funcs[parent][scriptAction]
		end
	end
	
	-- Function that defines the action 
	-- for the apply button in the main
	-- interface of the hotkey manager.
	function btnApply:action()
		
		-- Define the output location for the hotkey file.
		hKeyFile = fusion:MapPath("Profile:\\Fusion.hotkeys")
		
		-- Create a table from the scripted actions.
		stringToSave = hotkeysToString(hKeys)
--		stringToSave = eyeon.writefile(hKeys)
		
		stringToSave = string.gsub(stringToSave, "\\", [[\\]])
		-- set the location to output.
		fh = io.output(hKeyFile)
		
		-- Write the data to the .hotkey file
		fh:write(stringToSave)
		
		-- close it.
		fh:close()
		
		-- collect the garbage so that the fs is updated
		collectgarbage()
		
		-- load the hotkeys
		hKeyMan:LoadHotkeys()
		
		-- Hide the dialogs -- presumably we're done.
		iup.Hide(dlg)
		iup.Hide(dlg_hKeyAdd)
	end
	
	-- Hide the dialog... 
	function btnExit:action()
		iup.Hide(dlg)
	end
	
	-- Chunk wherein the AddHotkey dialog is created
	do
		-- Setup the dropdown..
		function setupDropdown()
			local count = 0
			local t = {}
			for k, v in pairs(hKeys) do
				count = count + 1
				t[count] = k
			end
			table.sort(t, function (a,b) return (b > a) end)
			for k, v in pairs(t) do
				list_category[tostring(k)] = v
			end
			
		end
		
		function buildHKeyBox(selection)
			if selection ~= hKeySelection then
				hKeySelection = selection
				keys = buildKeyList(selection)
				
				table.sort(keys, function (a,b) return (b > a) end)

				list_availHkeys["1"] = nil	-- clear the list
				for k,v in pairs(keys) do					
					list_availHkeys[tostring(k)] = v
				end
			end
		end
		
		function getModifierString()
			local modifiers = ""
			if chkShift.value == "ON" then
				modifiers = "SHIFT_"
			end
			if chkCtrl.value == "ON" then
				modifiers = modifiers.."CONTROL_"
			end
			if chkAlt.value == "ON" then
				modifiers = modifiers.."ALT_"
			end
			return modifiers
		end

		function buildKeyList(category)
			local exceptionList = {}
			local count = 0

			if category == "Global" then
				for k, v in pairs(hKeys) do
					for i, w in pairs(v) do
						count = count + 1
						exceptionList[count] = i
					end
				end
			elseif category == "View" or category == "GLViewer" then
				for k, v in pairs(hKeys) do
					if k == "View" or "GLViewer" then
						for i, w in pairs(v) do
							count = count + 1
							exceptionList[count] = i
						end
					end
				end
			elseif category == "Controls" or category =="Frame" then
				for k, v in pairs(hKeys) do
					if k == "Frame" or k == "Controls" then
						for i, w in pairs(v) do
							count = count + 1
							exceptionList[count] = i
						end
					end
				end
			else
				for k, v in pairs(hKeys[category]) do
					count = count + 1
					exceptionList[count] = k
				end
			end

			ret = {}
			count = 0
			
			local modifiers = getModifierString()

			for k, v in pairs(table_Hotkeys) do
				if not eyeon.isin(exceptionList, modifiers..v) then 
					count = count + 1
					ret[count] = v
				end
			end
			
			return ret
		end
		
		-- Create a list element that will be a dropdown box for the various
		-- key categories.
		-- see : http://www.tecgraf.puc-rio.br/iup/en/elem/iuplist.html
		list_category = iup.list{SIZE="150",DROPDOWN="YES"}
		
		-- Create a list to display the available hotkeys.
		list_availHkeys = iup.list{SIZE = "150x150"}
		
		-- Create checkboxes for key modifiers
		chkShift = iup.toggle{title="Shift"}
		chkCtrl  = iup.toggle{title="Ctrl"}
		chkAlt   = iup.toggle{title="Alt"}
		
		-- Create a button that will submit the name of the hotkey.
		btnSubmit = iup.button{title="OK", SIZE = "70x15"}
		
		-- Button for cancel..
		btnCancelAdd = iup.button{title = "Cancel",SIZE="70x15"}
		
		-- The dialog contains a vbox and an hbox.  What this means is that the interface
		-- items are aligned vertically, except for the Submit/Cancel buttons, which are aligned
		-- horizontally.
		
		-- Because there's been some weirdness with iup elements not resizing properly in the past,
		-- I've made the interface items statically sized, but I haven't defined the size of the
		-- interface.  IUP is smart enough to create a properly sized interface on its own, for the most
		-- part.
		dlg_hKeyAdd  = iup.dialog{
			iup.vbox{
				list_category, list_availHkeys,
				iup.hbox{chkShift, chkCtrl, chkAlt, expand="YES"},
				iup.fill{SIZE="10"},
				iup.hbox{btnSubmit, btnCancelAdd, title="Select Hotkey to Add"}
			},
			BGCOLOR="60 60 60", FGCOLOR = "200 200 200",
			TOPMOST = "NO",  RESIZE = "NO", MAXBOX = "NO"
		}

		-- Setup the dropdown using a function defined above..
		setupDropdown()
		
		-- When you change the category item that's selected, it should update the list_availHkeys
		-- to the new set for that particular grouping (seeing as we're keeping those hotkeys that
		-- are already used out of said list.
		-- So make an action for this item..
		function list_category:action()
			
			-- The value of the list is numeric.  Or rather a string that is always a number.  IUP 
			-- has some interesting idioms.
			-- To get that string / number (strumber), the syntax is list.value.  That strumber
			-- can then be used as an index in the list to get the associated string -- in this case
			-- the hotkey that we want to add.
			
			-- We keep track of this information just so that we're not constantly re-filling the 
			-- list_availHkeys..
			drop_id = list_category[list_category.value]
			
			-- We have a prev_drop_id set up..
			if drop_id ~= prev_drop_id then
				-- if it's not the same, rebuild the key.
				buildHKeyBox(drop_id)
			end
			
			-- Set the prev_drop_id variable.
			prev_drop_id = drop_id
		end

		function setListCategory(category)
			local i = 1
			while list_category[tostring(i)] do
				if list_category[tostring(i)] == category then
					list_category.value = tostring(i)
					break
				end
				i = i + 1
			end
		end
		
		-- rebuild the list when modifiers are toggled
		function chkShift:action()
			local sel = hKeySelection
			hKeySelection = ""
			buildHKeyBox(sel)
		end
		function chkCtrl:action()
			local sel = hKeySelection
			hKeySelection = ""
			buildHKeyBox(sel)
		end
		function chkAlt:action()
			local sel = hKeySelection
			hKeySelection = ""
			buildHKeyBox(sel)
		end
		
		-- Here's the action to submit the new hotkey
		function btnSubmit:action()
			selItem = getModifierString()..list_availHkeys[list_availHkeys.value]
			drop_id = list_category[list_category.value]
			
			hKeys[drop_id][selItem] = ""
			populateTree(true, selItem,drop_id)
			
			-- select the item we just added
			local i = 1
			while tree["NAME"..tostring(i)] do
				if tree["NAME"..tostring(i)] == selItem then
					tree["MARKED"..tostring(i)] = "YES"
					tree.value = tostring(i)
					break
				end
				i = i + 1
			end
			
			iup.Hide(dlg_hKeyAdd)
			list_category.value = "1"
			list_category:action()
			list_availHkeys.value = "1"
		end
		
		function btnCancelAdd:action()
			iup.Hide(dlg_hKeyAdd)
		end
		
	end
	
	-- design the dialog box.
	dlg = iup.dialog{
		iup.vbox{
			iup.hbox{
				iup.vbox{tree},
				iup.vbox{listboxScripts, textScript, btnSetScript},
			},
			iup.hbox{
				iup.fill{},
				btnApply, btnExit,
				iup.fill{},
			}
		},
		SIZE="450x250", GAP="10", MARGIN="5x5",
		menu=menu, title="Hotkey Manager",
		BGCOLOR="60 60 60", FGCOLOR="200 200 200",
		TOPMOST="NO", RESIZE="YES", MAXBOX="NO"
	}
	
	
	iup.SetAttribute(dlg, "NATIVEPARENT", touserdata(fusion:GetMainWindow()))
	
	-- show it
	dlg:showxy(iup.CENTER,iup.CENTER)
	
	-- Initializes the tree items.
	do
		tree.name = "Hotkeys"
		populateTree()
		gatherUnknownFunctions()
		ConcatenateLists()
		ChangeListsToAlphabetical()
		tree.ctrl = "NO"
		tree.shift = "NO"
		tree.addexpanded = "YES"
		tree.redraw = "YES"
	end
	
	
	-- Throw it into the Main loop.
	iup.MainLoop()
	
	-- destroy the dialogs after the mainloop is exited..
	dlg_hKeyAdd:destroy()
	dlg:destroy()
	dlg_Help:destroy()
end
